/**
 * \file: VideoChannel.cpp
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * <brief description>.
 * <detailed description>
 * \component: Android Auto
 *
 * \author: Y. Nakamura / ADITJ / ynakamura@jp.adit-jv.com
 *
 * \copyright (c) 2014 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 * \see <related items>
 *
 * \history
 *
 ***********************************************************************/

// to use std::to_string
#include <string>
#include <sys/types.h>
#include <unistd.h>
#include <pthread.h>
#include <signal.h>
#include <time.h>
#include <sys/prctl.h>
#include <adit_logging.h>
#include <errno.h>
#include <chrono>

#include "VideoChannel.h"
#include "DemoFactory.h"
#include "AutoSmoketest.h"
#include "libtestbedapp.h"
#include "Testbed.h"

LOG_IMPORT_CONTEXT(demo)

namespace adit { namespace aauto {


VideoChannel::VideoChannel(::shared_ptr<GalReceiver> inReceiver, uint8_t inSessionId, bool inAutoStart,
                           t_ilm_layer inLayerID, t_ilm_surface inSurfaceID, IDynamicConfiguration& inConfig) :
    receiver(inReceiver), sessionId(inSessionId), autoStart(inAutoStart),
    layerID(inLayerID), surfaceID(inSurfaceID), mConfig(&inConfig)
{
    endpoint = nullptr;
    mStop = false;
}

VideoChannel::~VideoChannel()
{
    if (!mStop) {
        shutdown();
    }
    endpoint = nullptr;
}

void VideoChannel::shutdown()
{
    LOG_INFO((demo, "video channel shut down"));
    mStop = true;
    mConditonVar.notify_one();

    if(endpoint != nullptr) {
        endpoint->shutdown();
    }
}

bool VideoChannel::Initialize()
{
    /* This pointer is deleted at shared_ptr in impl */
    if(Testbed::instance().getTestbedMode())
    {
        autoStart = false;
        Testbed::instance().setVideoChannel(this);
    }

    auto creator = DemoFactory::instance().getCreator
            <AditVideoSink, VideoSinkCreatorFn>(mConfig->GetItem("adit-video-sink", ""));
    if (creator != nullptr)
    {
        endpoint = creator(sessionId, receiver->messageRouter(), autoStart);
    }
    else
    {
        LOG_ERROR((demo, "VideoChannel::Initialize()  failed to get creator object"));
        return false;
    }
 
    if(endpoint == nullptr)
    {
        LOG_ERROR((demo, "VideoChannel::Initialize()  failed to get output of creator"));
        return false;
    }

    /* get list of VideoSink configuration values */
    std::list<std::string> video_sink = mConfig->GetItems("video-sink");
    for (auto node : video_sink)
    {
        std::string key;
        std::string value;
        /* get key and respective value from node entry */
        mConfig->GetNodeItem(node, key, value);

        if (0 == key.compare("gstreamer-video-pipeline")) {
#ifdef IMX_DEMO     // #ifdef should be handled by *.cfg
            value += " layer-id=";
            value += std::to_string(layerID);
            value += " surface-id=";
            value += std::to_string(surfaceID);
#else
            value += " surface-id=";
            value += std::to_string(surfaceID);
#endif
            LOG_INFO((demo, "Video pipeline: %s",value.c_str()));
        }

        /* Configuration for VideoSink Endpoint */
        endpoint->setConfigItem(key, value);
    }
    
    endpoint->registerCallbacks(this);
    
    if(!endpoint->init())
    {
        LOG_ERROR((demo, "VideoChannel::Initialize()  failed to initialize videoSink endpoint"));
        return false;
    }
    // The argument of API is done const_cast because GalReceiver requires non const pointer.
    if(!receiver->registerService(const_cast<AditVideoSink*>(endpoint.get())))
    {
        LOG_ERROR((demo, "VideoChannel::Initialize()  failed to register videoSink endpoint to receiver"));
        return false;
    }
    
    return true;
}

void VideoChannel::setupCallback(int inMediaCodecType)
{
    LOGD_DEBUG((demo, "VideoChannel::setupCallback()  inMediaCodecType = %d", inMediaCodecType));
}

void VideoChannel::playbackStartCallback()
{
    //There would be a logic here when playback starts.
    LOGD_DEBUG((demo, "VideoChannel::playbackstartcallback()"));
}

void VideoChannel::playbackFirstFrameRenderedCallback()
{
    // This is invoked when the video playback actually starts or rather when the first
    // frame is ready to be shown.

    LOGD_DEBUG((demo, "VideoChannel::playbackFirstFrameRenderedCallback()"));
    AutoSmoketest::instance().setTestError(VIDEOFRAMERENDERED);
}

void VideoChannel::playbackStopCallback()
{
    //There would be a logic here when playback stops.
    LOGD_DEBUG((demo, "VideoChannel::playbackstopcallback()"));
}

void* VideoChannel::delayedSwitchBack(void* inData)
{
    auto me = static_cast<VideoChannel*>(inData);
    if (me == nullptr)
        return nullptr;

    // set thread name
    prctl(PR_SET_NAME, "VideoChDelay", 0, 0, 0);

    // don't do this in production code !!!
    int msec = 10000;

    LOGD_DEBUG((demo, "VideoChannel::delayedSwitchBack()  switch back to AAUTO in %d seconds", (msec / 1000)));
    std::unique_lock<std::mutex> guard(me->mConditionMutex);
    me->mConditonVar.wait_for(guard, std::chrono::milliseconds(msec));

    // check if we are stopped while sleeping
    if (!me->mStop)
    {
        if (me->endpoint != nullptr)
        {
            LOGD_DEBUG((demo, "VideoChannel::delayedSwitchBack()  set VideoFocus to VIDEO_FOCUS_PROJECTED(%d)", (int)VIDEO_FOCUS_PROJECTED));
            me->endpoint->setVideoFocus(VIDEO_FOCUS_PROJECTED, false);
        }
        else
        {
            LOG_WARN((demo, "VideoChannel::delayedSwitchBack()  endpoint is null, assume that VideoChannel was shut downed."));
        }
    }
    else
    {
        LOGD_DEBUG((demo, "VideoChannel::delayedSwitchBack()  Stop triggered."));
    }

    pthread_exit(nullptr);
}

void VideoChannel::sourceVideoConfigCallback(int inWidth, int inHeight,
        int inUIResWidth, int inUIResHeight)
{
    //There would be a logic here when receives video configuration.
    LOGD_DEBUG((demo, "VideoChannel::sourceVideoConfigCallback() ,"\
                "inWidth = %d, inHeight = %d, inUIResWidth = %d,"\
                " inUIResHeight = %d", inWidth, inHeight, inUIResWidth, inUIResHeight));
}

void VideoChannel::videoFocusCallback(int inFocus, int inReason)
{
    // In this example, video focus is granted unconditionally. Ordinarily, there would
    // be logic here to conditionally grant video focus.

    LOGD_DEBUG((demo, "VideoChannel::videoFocuscallback()"));

    if(Testbed::instance().getTestbedMode())
    {
        if(inFocus == VIDEO_FOCUS_NATIVE)
        {
            LOG_INFO((demo, "VideoChannel::videoFocuscallback()  Switch AAUTO to background, native HMI to foreground"));
            endpoint->setVideoFocus(inFocus, true);
        }
        else if(inFocus == VIDEO_FOCUS_PROJECTED)
        {
            LOG_INFO((demo, "VideoChannel::videoFocuscallback()  Switch native HMI to background, AAUTO to foreground"));

            // Use to switch to foreground without touching icon in ADIT HMI
            Testbed::instance().adit_testbed_foreground();
        }
    }
    else
    {
        endpoint->setVideoFocus(inFocus, true);

        if (inFocus == VIDEO_FOCUS_NATIVE)
        {
            pthread_attr_t attr;
            int err = -1;
            if (0 == (err = pthread_attr_init(&attr)))
            {
                /* Mark the thread identified by threadId as detached.
                 * When a detached thread terminates, its resources are automatically released back to the system.
                 * There is no need for pthread_join(). */
                if (0 == (err = pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_DETACHED)))
                {
                    // switch back to mobile after some time
                    pthread_t threadId;
                    if (0 != (err = pthread_create(&threadId, &attr, delayedSwitchBack, (void*)this)))
                    {
                        LOG_WARN((demo, "VideoChannel::videoFocuscallback()  pthread_create() failed with err: %d, errno: %d", err, errno));
                    }
                }
                else
                {
                    LOG_WARN((demo, "VideoChannel::videoFocuscallback()  pthread_attr_setdetachstate() failed with err: %d, errno: %d", err, errno));
                }

                pthread_attr_destroy(&attr);
            }
            else
            {
                LOG_WARN((demo, "VideoChannel::videoFocuscallback()  pthread_attr_init() failed with err: %d, errno: %d", err, errno));
            }
        }
    }
}

void VideoChannel::notifyErrorCallback(aautoErrorCodes inErrorCode)
{
    LOG_ERROR((demo, "VideoChannel::notifyErrorCallback()  ErrorCode: %d", inErrorCode));
    AutoSmoketest::instance().setTestError(GSTINTERNAL);
}

} } /* namespace adit { namespace aauto */
